<?php

namespace App\Http\Controllers\Branch;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Model\WarehouseModel;
use DataTables;
use Validator;
use Carbon\Carbon;

class WarehouseController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        if($request->ajax()){
            $data = WarehouseModel::where('status', 1)
                        ->with('branch')
                        ->where('user_id', \Auth::id())
                        ->where(function($q) use ($request){
                            if($request->address){
                                $q->where('location_name', 'like', '%' . $request->address . '%');
                            }
                            if($request->description){
                                $q->where('description', 'like', '%' . $request->description . '%');
                            }
                            if($request->warehouse){
                                $q->where('warehouse_name', 'like', '%' . $request->warehouse . '%');
                            }
                            if($request->created_at){
                                $q->whereDate('created_at', $request->created_at);
                            }
                        })
                        ->selectRaw('warehouse_name as name, description, id, created_at, main, lat, lng, branch_id, location_name, status, main')
                        ->orderBy('id', 'desc')
                        ->paginate($request->record , ['*'], 'page', str_replace(',', '', $request->page));
            return DataTables::of($data->items())
                    ->with([
                        "activePage" => $request->page,
                        "records" => $request->record,
                        "recordsTotal" => $data->total(),
                        "recordsFiltered" => $data->total(),
                        'totalPage' => $data->lastPage()
                    ])
                    ->addColumn('action', function($action){
                        return '<i href="javascript:;" data-url = "' . route("branch-warehouse.update", $action->id) . '" data-name="'.$action->name.'" data-description="'.$action->description.'" data-id="'.$action->id.'"  data-branch_id="'.$action->branch_id.'" data-address="'.$action->location_name.'" class="fa fa-pencil btn btn-sm btn-warning" title="Edit"></i>'.' '.
                                '<i href="javascript:;" data-url = "' . route("branch-warehouse.destroy", $action->id) . '" class="fa fa-trash btn btn-sm btn-danger" title="Delete"></i>';
                    })
                    ->editColumn('created_at', function($action){
                        return Carbon::parse($action->created_at)->format('d-m-Y');
                    })
                    ->addColumn('branch', function($action){
                        return $action->branch?$action->branch->name:'';
                    })
                    ->addIndexColumn()
                    ->rawColumns(['action', 'branch'])
                    ->make(true);
        }
        return view('branch.warehouse.index');
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|min:1|max:25|string|unique:nso007_warehouse,warehouse_name',
            'address' => 'required|min:1|max:25|string',
            'description' => 'nullable|string|max:255',
        ]);

        $messages = [];
        if($validator->fails()){
            foreach ($validator->errors()->getMessages() as $item) {
                $messages[] .= $item[0];
            }
            return response()->json([$messages, 401]);
        }else{
            $warehouse = new WarehouseModel;
            $warehouse->warehouse_name = $request->name;
            $warehouse->location_name = $request->address;
            $warehouse->description = $request->description;
            $warehouse->branch_id = \Auth::id();
            $warehouse->user_id = \Auth::id();
            $warehouse->save();
            return response()->json([$warehouse, 200]);
        }
    }

    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|min:1|max:25|string|unique:nso007_warehouse,warehouse_name,'.$id,
            // 'branch' => 'required|exists:nso007_hr_branch,id|unique:nso007_warehouse,branch_id,'.$id,
            'address' => 'required|min:1|max:25|string',
            'description' => 'nullable|string|max:255',
        ]);

        $messages = [];
        if($validator->fails()){
            foreach ($validator->errors()->getMessages() as $item) {
                $messages[] .= $item[0];
            }
            return response()->json([$messages, 401]);
        }else{
            $warehouse = WarehouseModel::find($id);
            $warehouse->warehouse_name = $request->name;
            $warehouse->location_name = $request->address;
            $warehouse->description = $request->description;
            $warehouse->branch_id = \Auth::id();
            $warehouse->user_id = \Auth::id();
            $warehouse->update();
            return response()->json([$warehouse, 200]);
        }

    }

    public function destroy($id)
    {
        WarehouseModel::findOrFail($id)->update(['status' => 0]);
        return response()->json(['', 200]);
    }
}
