<?php


    namespace App\Http\Controllers\Administrator\stock_inventory;

    use App\Fun\Fun;
    use App\Http\Controllers\Controller;
    use App\Model\Accounting\AccountChartModel;
    use App\Model\Accounting\JournalModel;
    use App\Model\CostOfSoldModel;
    use App\Model\PaymentDetailModel;
    use App\Model\PaymentModel;
    use App\Model\ProductUnitModel;
    use App\Model\PurchaseDetailsModel;
    use App\Model\PurchaseModel;
    use App\Model\SaleDetailsModel;
    use App\Model\SaleModel;
    use App\Model\stock_inventory\ItemRequestModel;
    use App\Model\StockModel;
    use App\Model\SupplierModel;
    use App\Model\WarehouseModel;
    use App\User;
    use Auth;
    use Carbon\Carbon;
    use DataTables;
    use DB;
    use Illuminate\Database\Eloquent\Model;
    use Illuminate\Http\Request;
    use Illuminate\Validation\ValidationException;


    class PurchaseItemController extends Controller
    {

        function __construct()
        {
            $this->middleware('permission:New Purchase', ['only' => ['create', 'store']]);
        }

        public function index(Request $request)
        {
            if ($request->ajax()) {
                $if_member = ifMember();

                $data = PurchaseModel:: Join('users', 'nso007_purchases.user_id', 'users.id')
                    ->leftJoin('nso007_payment', 'nso007_purchases.id', 'nso007_payment.invoice_id')
                    ->selectRaw(
                        'invoice_number, nso007_purchases.id, nso007_payment.id as payment_id,
                    purchase_date, nso007_purchases.payment_status, paid_amount as paid_usd,
                    discount, users.name,nso007_purchases.member_id')
                    ->where('nso007_purchases.status', '!=', 'pending')
                    ->where('nso007_purchases.confirm_receive', '!=', 'pending')
                    ->where('nso007_payment.type', 'purchase')
                    ->where(function ($q) use ($if_member) {
                        if ($if_member) {
                            $q->where('nso007_purchases.member_id', Auth::id());
                            $q->orwhere('nso007_purchases.request_to', Auth::id());
                        } else {
                            $q->where('nso007_purchases.branch_id', FindBranchID());
                        }
                    })
                    ->where(function ($q) use ($request) {
                        if ($request->invoice_number) {
                            $q->where('invoice_number', 'like', '%' . $request->invoice_number . '%');
                        }
                        if ($request->payment_status) {
                            $q->where('nso007_purchases.payment_status', 'like', '%' . $request->payment_status . '%');
                        }
                        if ($request->created_at) {
                            $q->whereDate('purchase_date', $request->created_at);
                        }
                    })
                    ->groupBy('nso007_purchases.invoice_number')
                    ->orderBy('nso007_purchases.id', 'DESC')
                    ->paginate($request->record, ['*'], 'page', str_replace(',', '', $request->page));

                return DataTables::of($data->items())
                    ->with([
                        "activePage" => $request->page,
                        "records" => $request->record,
                        "recordsTotal" => $data->total(),
                        "recordsFiltered" => $data->total(),
                        'totalPage' => $data->lastPage()
                    ])
                    ->addColumn('action', function ($action) {
                        $delete = '<button data-href="' . route('purchases-item-delete', $action->id) . '" class="btn btn-danger fa fa-trash btn_delete"></button>';
                        $sale = SaleModel::where('purchase_request_id', $action->id)->first();
                        $payment_status = $action->member_id === Auth::id() ? 'paid' : $action->payment_status;
                        if ($sale) {
                            $payment_status = 'paid'; //make invoice cant paid if have in wholesale
                        }
                        return '<i href="javascript:;" data-url = "' . route("purchases-item.show", $action->id) . '"
                                data-payment_url = "' . route("payment.edit", $action->payment_id) . '"
                                data-payment_status = "' . $payment_status . '" class="fa fa-eye btn btn-sm btn-primary" title="Show"></i>
                                ';
                    })
                    ->editColumn('payment_status', function ($action) {
                        if ($action->payment_status)
                            return textStatus($action->payment_status);
                    })
                    ->editColumn('paid_usd', function ($action) {
                        return number_format($action->paid_usd, 2) . '$';
                    })
                    ->addColumn('total', function ($action) {
                        $total = \DB::table('nso007_payment')
                            ->where('invoice_id', $action->id)
                            ->where('type', 'purchase')
                            ->first()->payment_amount;
                        return number_format($total, 2) . '$';
                    })
                    ->addColumn('punishment', function ($action) {
                        $payment = \DB::table('nso007_payment')
                            ->where('invoice_id', $action->id)
                            ->where('type', 'purchase')
                            ->first();
                        $punish = $payment ? $payment->punishment : 0;
                        $payment_amount = $payment->payment_amount;
                        $punish = ($payment_amount * $punish) / 100;
                        $action->punish = $punish;
                        return '$' . number_format($punish, 2);
                    })
                    ->addColumn('remain_amount', function ($action) {
                        $total = \DB::table('nso007_payment')
                            ->where('invoice_id', $action->id)
                            ->where('type', 'purchase')
                            ->first()->payment_amount;
                        $total = $total + $action->punish - $action->paid_usd;
                        return number_format($total, 2) . '$';
                    })
                    ->addIndexColumn()
                    ->rawColumns(['action', 'payment_status', 'punishment'])
                    ->make(true);
            }
            return view('administrator.stock-inventory.purchases.index');
        }

        public function create(Request $request)
        {
            Fun::lang();
            $supplier = SupplierModel::pluck('name', 'id');
            $acc_chart = AccountChartModel::whereIn('id', ['11001', '11002'])->pluck((if_kh() ? 'acc_namekh' : 'acc_name'), 'id');
            $member_po = $request['poreq'] ? decrypt($request['poreq']) : '';
            $purchase_detail = PurchaseDetailsModel::where('purchase_id', $member_po)->get();
            return view('administrator.stock-inventory.purchases.item.create', compact('supplier', 'acc_chart', 'purchase_detail', 'member_po'));
        }

        public function show($id)
        {
            $master = PurchaseModel::find($id);
            return view('administrator.stock-inventory.purchases.item.show', compact('master'));
        }

        public function edit($id)
        {
            $supplier = SupplierModel::pluck('name', 'id');
            $acc_chart = AccountChartModel::whereIn('id', ['11001', '11002'])->pluck((if_kh() ? 'acc_namekh' : 'acc_name'), 'id');
            $purchase = PurchaseModel::find($id);
            $purchase_detail = PurchaseDetailsModel::where('purchase_id', $id)->get();
            return view('administrator.stock-inventory.purchases.item.edit', compact('purchase', 'supplier', 'acc_chart', 'purchase_detail'));
        }

        public function store(Request $request)
        {
            return $this->update($request, null);
        }

        public function update(Request $request, $id)
        {
            $check = !ifMember() ? 'required' : 'nullable';
            $check1 = $request['payment_type'] === 'pay_now' ? 'required' : 'nullable';
            $this->validate($request, [
                'purchase_date' => 'required|date',
                'supplier_id' => $check,
                'invoice_number' => $check,
                'payment_type' => $check,
                'payment_method' => $check1,
                'paid_usd' => $check1,
                'product_unit_id.*' => 'required|exists:nso007_product_unit,id',
                'qty.*' => 'required',
                'cost.*' => $check,
                'shelf.*' => $check,
                'description' => 'nullable|max:255|string',
            ]);
            $reference = '';
            $purchase = PurchaseModel::find($id);
            if ($request['reference']) {
                $files = $request['reference'];
                $original_name = date('ymdhis') . $request['reference']->getClientOriginalName();
                $destinationPath = 'images/purchase';
                $files->move($destinationPath, $original_name);
                $reference = $destinationPath . '/' . $original_name;
                if ($purchase->reference ?? null) {
                    if (file_exists(public_path() . '/' . $purchase->reference)) {
                        unlink(public_path() . '/' . $purchase->reference);
                    }
                }
            }
            if ($request['payment_type'] === 'pay_later') {
                $payment_method = '21001';
            } else {
                $payment_method = $request['sub_payment_method'] ?? $request['payment_method'];
            }
            $data = $request->all();
            $data['user_id'] = Auth::id();
            $data['branch_id'] = ifMember() ? 0 : mainBranch()->id;
            $data['rate'] = currencyExchange();
            $data['year'] = date('Y');
            $data['payment_status'] = $request['paid_usd'] >= $request['total_as_dollar'] ? 'paid' : ($request['paid_usd'] > 0 ? 'partial' : 'due');
            $data['payment_method'] = $payment_method;
            $data['reference'] = $reference;
            $data['grand_total'] = $request['total_as_dollar'];
            $data['invoice_number'] = $request['invoice_number'] ?? refNumber(Auth::id(), 'purchase', 'PU');
            $data['purchase_date'] = sDate($request['purchase_date']);
            if (ifMember()) {
                $po = PurchaseModel::find($request['member_po']);
                if ($po) {
                    $po->status = 'processing';
                    $po->update();
                    $data['member_purchase_id'] = $po->member_purchase_id ?: $request['member_po'];
                    // payment for last request
                    $payment = new PaymentModel();
                    $payment->user_id = Auth::id();
                    $payment->payment_invoice = $po->invoice_number;
                    $payment->invoice_id = $po->id;
                    $payment->payment_date = date('Y-m-d');
                    $payment->payment_amount = $po->grand_total;
                    $payment->paid_amount = $po->paid_usd;
                    $payment->payment_status = $po->payment_status;
                    $payment->year = date('Y');
                    $payment->status = 1;
                    $payment->type = 'purchase';
                    $payment->save();
                } else {
                    $data['member_purchase_id'] = $request['member_po'];
                }
                $data['request_to'] = Auth::user()->leader_id;
                $data['request_to_type'] = myParent()->type;
                $data['member_id'] = Auth::id();
                $data['delivery_to_member'] = $po->delivery_to_member ?? Auth::id();
            }

            $purchase = PurchaseModel::updateOrCreate(['id' => $id], $data);
            PurchaseDetailsModel::where('purchase_id', $id)->delete();
            foreach ($request['product_unit_id'] as $key => $item) {
                $purchase_detail = new PurchaseDetailsModel;
                $purchase_detail->purchase_id = $purchase->id;
                $purchase_detail->product_unit_id = $item;
                $purchase_detail->qty = $request['qty'][$key];
                $purchase_detail->deliver_qty = $request['qty'][$key];
                $purchase_detail->cost = $request['cost'][$key];
                $purchase_detail->dis = $request['dis'][$key];
                if ($request['shelf']) {
                    $purchase_detail->shelf = $request['shelf'][$key];
                }
                $purchase_detail->save();
            }
            session()->put('success', 'Record inserted successfully');
            return response()->json(['status' => 'ok', 'route' => route('purchases-item-approval') . '?show=' . (ifMember()?('p'):($purchase->branch_id ? 'p' : 'c'))]);
        }

        public function destroy($id)
        {
            PurchaseModel::find($id)->update(['status' => 0]);
            return response()->json('');
        }

        public function getApproval(Request $request)
        {
            if ($request->ajax()) {
                $user_id = Auth::user()->id;
                $data = PurchaseModel::where(function ($q) {
//                        $q->whereIn('status', ['pending', 'processing', 'reject']);
//                        if (ifMember())
//                            $q->orwhere('confirm_receive', 'pending');
                    })
                    ->where(function ($q) use ($request,$user_id) {
                        if (ifMember()) {// for member
                            if ($request['show']==='p') {
                                //show Auth Purchase
                                $q->where('member_id', $user_id);
                            }else{
                                // show purchase from member
                                $q->where('request_to', $user_id);
                                $q->where('request_to_type', 'member');
                            }
                        }else{// for company
                            if ($request['show']==='p') {
                                //show Auth Purchase
                                $q->where('branch_id', FindBranchID());
                            }else{
                                // show purchase from member
                                $q->where('request_to', $user_id);
                                $q->where('request_to_type', 'branch');
                            }
                        }

                    })
                    ->where(function ($q) use ($request) {
                        if ($request['from_date']) {
                            $q->whereDate('purchase_date', '>=', sDate($request['from_date']));
                        }
                        if ($request['to_date']) {
                            $q->whereDate('purchase_date', '<=', sDate($request['to_date']));
                        }
                        if ($request['payment_status']) {
                            $q->where('payment_status', $request['payment_status']);
                        }
                        if ($request['user_id']) {
                            $q->where('user_id', $request['user_id']);
                        }
                        if ($request['supplier_id']) {
                            $q->where('supplier_id', $request['supplier_id']);
                        }
                    })
                    ->orderBy('id', 'desc');
                $count = $data->count();
                return DataTables::of($data->take(20))
                    ->with([
                        "recordsTotal" => $count,
                        "recordsFiltered" => $count,
                    ])
                    ->addColumn('action', function ($action) {
                        $if_requester = $action->member_id == Auth::id();
                        $if_admin = $action->request_to_type === 'member' ? ($action->request_to == Auth::id()) : ($action->request_to == FindBranchID());
                        $btn = '<a data-href = "' . route("purchases-item.show", $action->id) . '" class="fa fa-eye btn btn-sm btn-info getHtmlFormShow" data-width="900px" title="Show"></a>';

                        //requester can edit when admin not approve yet
                        if ($if_requester && $action->status === 'pending') {
                            $btn .= ' <a href="' . route("purchases-item.edit", $action->id) . '" class="fa fa-pencil btn btn-sm btn-warning"></a>';
                            $btn .= ' <a data-href="' . route("purchases-item-delete", $action->id) . '" class="fa fa-trash btn_delete btn btn-sm btn-danger" title="Delete"></a>';
                        } else {
                            // first member PO can confirm receive
                            if ($action->delivery_to_member === Auth::id() && $action->status === 'approved' && $action->confirm_receive === 'pending') {
                                $btn .= ' <a data-href="' . route("purchases-item-confirm-receive", $action->id) . '" class="btn_receive btn btn-sm btn-success fa fa-check-square-o"  title="Receive"></a>';
                            }
                        }
                        // admin can approve or reject
                        if ($if_admin || !ifMember()) {
                            if ($action->status === 'pending' && $action->confirm_receive === 'pending') {
                                if ($action->request_to) {
                                    $href = 'href = "' . route("wholesale.create") . '?poreq=' . encrypt($action->id) . '"';
                                    $class = '';
                                } else {
                                    $href = 'data-url = "' . route("purchases-item-approve", [$action->id, 'approved']) . '"';
                                    $class = 'btn-approve';
                                }
                                if (ifMember()) {
                                    if ((myParent()->type ?? 0) === 'member') {
                                        $request_href = route('purchases-item.create') . '?';
                                    } else {
                                        $request_href = route('request-stock.create') . '?if_approve=' . encrypt(0) . '&';
                                    }
                                    $make_request = '<a href="' . $request_href . 'poreq=' . encrypt($action->id) . '" class="fa fa-mail-reply btn btn-sm btn-warning ' . $class . '" title="Make Request"></a>';
                                } else {
                                    $make_request = '';
                                }
                                $btn .= ' <a  data-url = "' . route("purchases-item-approve", [$action->id, 'cancel']) . '"  class="fa fa-times btn btn-sm btn-danger" title="Cancel"></a>
                                        <a ' . $href . 'class="fa fa-check btn btn-sm btn-primary ' . $class . '" title="Approve"></a> ' . $make_request;
                            }
                        }
                        return $btn;
                    })
                    ->editColumn('payment_status', function ($action) {
                        return textStatus($action->payment_status);
                    })
                    ->editColumn('purchase_date', function ($action) {
                        return myDate($action->purchase_date);
                    })
                    ->editColumn('status', function ($action) {
                        return textStatus($action->status);
                    })
                    ->editColumn('confirm_receive', function ($action) {
                        return textStatus($action->confirm_receive);
                    })
                    ->editColumn('paid_usd', function ($action) {
                        return number_format($action->paid_usd, 2) . '$';
                    })
                    ->editColumn('grand_total', function ($action) {
                        return number_format($action->grand_total, 2) . '$';
                    })
                    ->addColumn('remain_amount', function ($action) {
                        return number_format($action->grand_total - $action->paid_usd, 2) . '$';
                    })
                    ->editColumn('member_id', function ($action) {
                        if ($action->member_id)
                            return findUser($action->member_id)->name ?? '';
                        return findUser($action->user_id)->name ?? '';
                    })
                    ->editColumn('request_to', function ($action) {
                        if ($action->request_to) {
                            if ($action->request_to_type === 'member')
                                return findUser($action->request_to)->name ?? '';
                            return findBranch(mainBranch()->id)->name ?? '';
                        }
                        return findSupplier($action->supplier_id)->name ?? '';
                    })
                    ->addIndexColumn()
                    ->rawColumns(['action', 'payment_status', 'status', 'confirm_receive'])
                    ->make(true);
            }
            $user = PurchaseModel::where(function ($q) {
                $q->where('request_to', FindBranchID());
                $q->orwhere('request_to', Auth::user()->id);
                $q->orwhere('branch_id', FindBranchID());
            })->pluck('user_id');
            $user = User::whereIn('id', $user)->pluck('name', 'id');
            return view('administrator.stock-inventory.purchases.item.approval', compact('user'));
        }

        public function updateApproval($id, $type, $if_receive = false)
        {
            $purchase = PurchaseModel::find($id);
            if ($type === 'approved') {
                $item_request = ItemRequestModel::where('member_purchase_id', $id)->first();
                if ($item_request) {
                    $item_request->confirm_status = 'received';
                    $item_request->update();
                }
                $purchase->status = 'approved';
                if ($if_receive || FindBranchID() === 1) {
                    $purchase->confirm_receive = 'received';
                    SaleModel::where('purchase_request_id', $id)->update(['confirm_receive' => 'received']);
                }
                $purchase->update();
                $paid = $purchase->paid_usd;
                $purchase_details = PurchaseDetailsModel::where('purchase_id', $id)->get();
                $total = 0;
                $j = JournalModel::selectRaw("Max(journal_tran_id) as last_num")->first();
                $arr_journal = [
                    'branch_id' => $purchase->branch_id,
                    'journal_by_supplier' => $purchase->supplier_id,
                    'journal_type' => 1,
                    'voucher_type' => 2,
                    'journal_transactiondate' => $purchase->purchase_date,
                    'journal_paydate' => $purchase->purchase_date,
                    'journal_currency' => 2,
                    'transaction_type' => 'Purchase',
                    'journal_referenceid' => $purchase->id,
                    'voucher_ref' => "V-" . generate_voucher_number(($j->last_num + 1), 8),
                    'journal_by' => \Auth::id(),
                    'journal_reference_number' => $purchase->invoice_number,
                    'journal_tran_id' => $j->last_num + 1,
                ];

                if ($purchase_details) {
                    foreach ($purchase_details as $key => $item) {
                        $own_total = ($item->qty * $item->cost) - (($item->qty * $item->cost) * ($item->dis / 100));
                        $product_unit = ProductUnitModel::find($item['product_unit_id']);
                        $qty = $purchase->request_to ? $item->deliver_qty : $item['qty'];

                        if ($purchase->delivery_to_member) {
                            adjStockMember($purchase->delivery_to_member, $item['product_unit_id'], $qty);
                        } else {
                            adjStockAdmin($purchase->branch_id, $item['shelf'], $item['product_unit_id'], $qty);
                        }

                        $product = getProductById($product_unit->product_id);
                        $total += $own_total;
                        //===================================== cost of sold ========================================================
                        $cost = CostOfSoldModel::where(['branch_id' => $purchase->branch_id, 'product_unit_id' => $item['product_unit_id']])->first();
                        $item_cost = $item['cost'];
                        $item_cost *= $product_unit->qty_per_unit;
                        $cost = $cost->cost ?? $item_cost;
                        $cost = ($cost + $item_cost) / 2;
                        CostOfSoldModel::UpdateOrcreate(['branch_id' => $purchase->branch_id, 'product_unit_id' => $item['product_unit_id']], [
                            'cost' => $cost
                        ]);
                        if ($key === 0) {
                            $last_id = 0;
                        } else {
                            $last_id = maxJournalId();
                        }
                        // create account chat if doesn't existed
                        createProductAccountChart($product);
                        $arr_journal['journal_parentid'] = $last_id;
                        $arr_journal['journal_acccode'] = $product->acc_inventory;
                        $arr_journal['journal_des'] = 'Stock inventory ' . ($product->product_name ?? '') . '(' . ($product->item_code ?? '') . ')';
                        $arr_journal['journal_debit'] = $own_total;
                        $arr_journal['journal_credit'] = 0;
                        addJournal($arr_journal);
                    }
                }

                $total = $total - ($total * ($purchase->discount / 100));

                // payment
                $payment = new PaymentModel();
                $payment->user_id = Auth::id();
                $payment->payment_invoice = 'PI' . date('YmdHis');;
                $payment->invoice_id = $purchase->id;
                $payment->supplier_id = $purchase->supplier_id;
                $payment->payment_date = date('Y-m-d');
                $payment->payment_amount = $total;
                $payment->paid_amount = $purchase->paid_usd >= $total ? $total : $purchase->paid_usd;
                $payment->payment_status = $purchase->paid_usd >= $total ? 'paid' : 'owe';
                $payment->year = date('Y');
                $payment->status = 1;
                $payment->type = 'purchase';
                $payment->save();

                // payment detail
                $payment_detail = new PaymentDetailModel();
                $payment_detail->payment_id = $payment->id;
                $payment_detail->paid_amount = $purchase->paid_usd >= $total ? $total : $purchase->paid_usd;
                $payment_detail->payment_date = date('Y-m-d');
                $payment_detail->status = 1;
                $payment_detail->save();

                /* ====== Accounting ====== */

                if ($paid >= $total) {
                    $payment_status = 'paid';
                } else {
                    if ($paid > 0) {
                        $payment_status = 'partial';
                    } else {
                        $payment_status = 'due';
                    }
                }

                if ($paid >= $total) {
                    $arr_journal['journal_type'] = 1;
                    $arr_journal['voucher_type'] = 2;
                    $arr_journal['journal_parentid'] = maxJournalId();
                    $arr_journal['journal_acccode'] = $purchase->payment_method;
                    $arr_journal['journal_des'] = 'Paid Stock inventory ' . '(' . $purchase->invoice_number . ')';
                    $arr_journal['journal_debit'] = 0;
                    $arr_journal['journal_credit'] = $total;

                    addJournal($arr_journal);
                } else {
                    if ($payment_status === 'partial') {
                        $arr_journal['journal_parentid'] = maxJournalId();
                        $arr_journal['journal_des'] = 'Paid Some Stock inventory ' . '(' . $purchase->invoice_number . ')';
                        $arr_journal['journal_debit'] = 0;
                        $arr_journal['journal_credit'] = $paid;
                        addJournal($arr_journal);

                        $arr_journal['journal_parentid'] = maxJournalId();
                        $arr_journal['journal_type'] = 2;
                        $arr_journal['voucher_type'] = 2;
                        $arr_journal['journal_acccode'] = '21001';
                        $arr_journal['journal_des'] = 'Paid Some inventory ' . '(' . $purchase->invoice_number . ')';
                        $arr_journal['journal_debit'] = 0;
                        $arr_journal['journal_credit'] = $total - $paid;
                        addJournal($arr_journal);
                    } else {
                        $arr_journal['journal_parentid'] = maxJournalId();
                        $arr_journal['journal_type'] = 2;
                        $arr_journal['voucher_type'] = 2;
                        $arr_journal['journal_acccode'] = '21001';
                        $arr_journal['journal_des'] = 'Payable Stock inventory ' . '(' . $purchase->invoice_number . ')';
                        $arr_journal['journal_debit'] = 0;
                        $arr_journal['journal_credit'] = $total;
                        addJournal($arr_journal);
                    }
                }
                /* ====== End Account ====== */
            } else {
                $purchase->status = 'reject';
                $purchase->update();

            }
            return response()->json(['route' => route('purchases-item.index')]);
        }

        public function delete($id)
        {
            $purchase = PurchaseModel::find($id);
            PurchaseDetailsModel::where('purchase_id', $id)->delete();
            $purchase->delete();
            return response()->json(['status' => 'ok', 'reload' => 1]);
        }

        public function confirm_receive(Request $request, $id)
        {
            if ($request['confirm']) {
                return $this->updateApproval($id, 'approved', true);
            } else {
                $master = PurchaseModel::find($id);
                return view('administrator.stock-inventory.purchases.item.show', compact('master'));
            }

        }

    }


